
/*  D80.C - Main File
 *  8080/8085 Disassembler - Copyright (C) 1993 by
 *		Jeffery L. Post
 *		22726 Benner Ave.
 *		Torrance, CA  90505
 *
 *  Version 1.0 - Sep 12, 1993 - Initial release, MSDOS version.
 *
 *		Placed in the public domain August, 1996 (JLP).
 */

#define	verno		1
#define	revno		0
#define	pmemsize	65536
#define	byte		unsigned char
#define	word		unsigned int
#define	tstop		32
#define	dstop		40
#define	astop		56

#include	<stdio.h>
#include	<ctype.h>
#include	<alloc.h>
#include	"d80.tbl"				/* get global tables */

/*  Global variables */

char	src[32], dst[32];			/* file name buffers				*/
char	linebuffer[128];			/* input line buffer				*/
FILE	*fp;							/* dos file struct				*/
int	hexflag = 0;				/* append hex flag				*/
int	binflag = 0;				/* binary file flag				*/
int	kcnt;							/* output char counter			*/
int	opcount;						/* bytes/opcode count			*/
word	himark;						/* highest valid data			*/
word	offset;						/* program counter offset		*/
byte	far *pmem;					/* program data pointer			*/
byte	far *pflg;					/* pointer to program flags	*/
													
main(int argc, char *argv[])
{
	char	c;
	word	count;

	printf("\nD80 8080/8085 Disassembler V %d.%d\n", verno, revno);
	printf("Copyright (C) 1993 by J. L. Post\n");
	if (argc < 2)
	{
		printf("\nUsage: d80 file [options]\n");
		printf("Option 'd' will include address and data in the");
		printf(" comment field.\n");
		printf("Option 'b' will read a binary file instead of ");
		printf("a hex file.\n");
		printf("Option 'x' will add a hex offset to the program code.\n\n");
		printf("Creates assembly source 'file.z80' from 'file.hex'");
		printf(" or 'file.bin'.\n");
		printf("Options may be entered freeform, ie:\n");
		printf("'db' 'd b' '-d -b x100' '-bdx100' are all acceptable.\n");
		printf("The 'x<hex number>' option must be entered last.\n");
		exit(0);
	}

	strcpy(src, argv[1]);
	strcat(src, ".hex");
	strcpy(dst, argv[1]);				/* init filenames */
	strcat(dst, ".d80");

	if (argc > 2)							/* test for options */
	{
		for (count=2; count<argc; count++)
		{
			while (c = toupper(*argv[count]++))
			{
				if (c == 'X')
				{
					offset = atox(argv[count]);
					break;
				}
				else if (c == 'D')
					hexflag = 1;
				else if (c == 'B')
				{
					binflag = 1;
					strcpy(src, argv[1]);
					strcat(src, ".bin");
				}
			}
		}
	}

	if (binflag)
		fp = fopen(src, "rb");
	else
		fp = fopen(src, "r");
	if (fp == NULL)
	{
		printf("* Source file '%s' not found *\n", src);
		exit(0);
	}

	if ((pmem = (byte far *) farmalloc(pmemsize)) == NULL)
	{
		printf("* Can't allocate program space *\n");
		exit(0);
	}

	if ((pflg = (byte far *) farmalloc(pmemsize)) == NULL)
	{
		printf("* Can't allocate program space *\n");
		exit(0);
	}

	printf("Initializing program spaces...");
	for (count=0xffff; count; --count)			/* fill code space with	*/
		pmem[count] = '\xff';						/* invalid data			*/
	pmem[0] = '\xff';
	for (count=0xffff; count; --count)			/* invalidate flags	 */
		pflg[count] = '\xff';
	pflg[0] = '\xff';

	pass1();					/* read file and find internal references */
	pass2();					/* disassemble to output file */
	pass3();					/* generate equ's for references */
	printf("\nDone");
}								/*  End of Main */

atox(char *str)
{
	char	c;
	word	i;

	i = 0;
	while (c = toupper(*str++))
	{
		if (isxdigit(c))
		{
			c = (c > '9') ? c - 0x37 : c & 0xf;
			i = (i << 4) | c;
		}
		else
			break;
	}
	return(i);
}

getcode(char *from, byte far *loc)
{
	byte	c, i;

	c = *from++ - 0x30;
	c = (c > 10) ? c - 7 : c;
	i = c << 4;
	c = *from++ - 0x30;
	c = (c > 10) ? c - 7 : c;
	*loc = i | c;
}

/***********************************************************************
	Pass one of disassembly. Read HEX file data into data array and
	examine opcodes for internal references to other memory locations.
	If such references are found, flag the referenced location so that
	a label can be generated in the output file during pass two.
************************************************************************/

pass1()
{
	word	i, l, pc, rel;
	byte	j, k;
	char	*inp;

	himark = 0;
	pc = offset;
	printf("\nreading %s\n", src);

/* read hex file and set up data array */

	if (binflag)
	{
		while (!feof(fp))
		{
			i = fread(linebuffer, 1, 128, fp);
			for (l=0; l<i; l++)
			{
				pmem[pc] = linebuffer[l];
				pflg[pc] = '\0';
				pc++;
				if ((pc & 0xff) == 0)
					printf("\r%04x", pc);
			}
			himark = pc;
		}
	}

	else
	{
		while (!feof(fp))						/* until end of file */
		{
			*linebuffer = '\0';				/* clear previous junk */
			fgets(linebuffer, 127, fp);	/* read one line */
			inp = linebuffer;					/* local copy of pointer */
			sscanf(inp, "%*c%2x%4x", &i, &pc);	/* get count and address */
			if (i == 0)							/* done if end of hex record */
				break;
			if (i > 64)
				error("invalid count in %s", linebuffer);
			for (l=0; l<i ; l++)
			{
				getcode(inp+9+l*2, &pmem[pc]);
				pflg[pc] = '\0';
				pc++;
				if ((pc & 0xff) == 0)
					printf("\r%04x", pc);
			}
			himark = (himark > pc) ? himark : pc;
		}
	}
	fclose(fp);						/* done reading input file */
	if (!himark)
		himark = 0xffff;
	printf("\rHighest location = %04x\n", himark);

/*	now find address references in array,
	but don't flag un-initialized data
*/

	printf("Pass 1 0000");

	for (i=offset; i<himark; )
	{
		l = i & 0xff;
		k = pmem[i];					/* get stored opcode */
		if (pflg[i] & 0x80)			/* ignore un-initialized data */
			i++;
		else
		{
			j = opttbl[k];				/* get option byte */
			if (j & 0x10)				/* if absolute memory reference */
			{
				pc = (pmem[i+1] & 0xff) | ((pmem[i+2] << 8) & 0xff00);
				pflg[pc] = (pflg[pc] & 0xf7) | 1;		/* flag reference */
			}
			i = i + (opttbl[k] & 3) + 1;		/* update location pointer */
		}
		if ((i & 0xff) < l)
			printf("\rPass 1 %04x", i & 0xff00);
	}
	printf("\rPass 1 %04x", himark);
}														/*  End of Pass 1 */

/************************************************************************
	Pass two of disassembly. Rescan data array and generate output file.
	Generate label if location flagged as referenced by some other opcode.
	If un-initialized data is encountered, ignore it but set skip flag so
	that an ORG statement can be generated when the next initialized data
	is found.
************************************************************************/

pass2()
{
	char	c;
	int	skip = 1, nl = 0;
	word	i, l, pc, q;
	byte	j, k;

	k = 1;
	j = 0xef;

	printf("\nPass 2 0000");

	if ((fp = fopen(dst, "w")) == NULL)
	{
		printf("* Can't open file %s *", dst);
		exit(0);
	}
	fprintf(fp, ";\n;  8080/8085 Disassembly of %s", src);

/*  Generate output file */

	for (i=offset; i<himark; )
	{
		l = i & 0xff;
		if (pflg[i] & 0x80)
		{
			i++;					/* ignore un-initialized data */
			skip++;				/* flag skip for ORG statement */
			j = -1;
		}

/*	If previous data was an unconditional transfer, AND current
	location is not referenced by some other opcode, AND current
	byte is 0 or 0FFH, then treat this byte as un-initialized data.
*/

		else if ((j & 0x10) && ((pflg[i] & 1) == 0) &&
				((pmem[i] == 0) || (pmem[i] == 0xff)))
		{
			pflg[i] = '\xff';			/* flag as uninitialized data */
			i++;
			skip++;
		}

/*	If previous opcode was 0 or 0FFH, AND current location is not
	referenced, AND current opcode is 0 or 0FFH, un-initialize it.
*/

		else if ((k == 0 || k == 0xff) && ((pflg[i] & 1) == 0) &&
				(pmem[i] == 0 || pmem[i] == 0xff))
		{
			pflg[i] = '\xff';			/* flag as uninitialized data */
			i++;
			skip++;
			fprintf(fp, "\t; data truncated");
			j = -1;
		}

		else
		{
			if (skip)					/* insert ORG statement */
			{
				if (!nl)
					fprintf(fp, "\n;");
				fprintf(fp, "\n\torg\t");
				puthex(i);
				fprintf(fp, "\n;");
				skip = 0;				/* reset skip flag */
			}
			k = pmem[i];						/* get opcode */
			j = opttbl[k];						/* get options */
			if (j & 8)
				fprintf(fp, "\n;");				/* if invalid opcode */

/* add label to output line if current location marked as referenced */

			if (pflg[i] & 1)
				fprintf(fp, "\nX%04x:\t", i);
			else
				fprintf(fp, "\n\t");

			kcnt = 8;
			opcount = 1;
			doopcode(mnemtbl[k].mcode);	/* output mnemonic */

	/* Generate operands */

			switch(j & 0x7f)
			{
				case 0x00:					/* single byte - no options */
					break;

				case 0x05:					/* 2 byte immediate data */
					puthex(pmem[i + 1]);
					splitcheck(i + 1);
					break;

				case 0x08:					/* invalid opcode */
					puthex(k);
					break;

				case 0x06:
				case 0x16:					/* 3 byte immediate data */
					q = (pmem[i+2] << 8) | pmem[i+1];
					kcnt += fprintf(fp, "X%04x", q);
					splitcheck(i+1);
					splitcheck(i+2);
					break;

				default:
					printf("\nOOPS! - optable entry = 0x%02x, opcode = 0x%02x",
						j, k);
					break;
			}

			if (hexflag)
			{
				do
				{	putc('\t', fp);
					kcnt = (kcnt + 8) & 0x78;
				} while (kcnt < tstop);

				kcnt += fprintf(fp, "; %04x ", i);
				q = (opttbl[k] & 3) + opcount;
				for (c=0; c<q; c++)
					kcnt += fprintf(fp, " %02x", pmem[i + c]);
				do
				{	putc('\t', fp);
					kcnt = (kcnt + 8) & 0x78;
				} while (kcnt < astop);

				for (c=0; c<q; c++)
					putc(ascii(pmem[i + c]), fp);
			}

			nl = 0;
			if (j & 0x88)					/* if unconditional transfer */
			{
				fprintf(fp, "\n;");		/* or invalid opcode */
				nl++;
			}
			i = i + (opttbl[k] & 3) + opcount;	/* update location counter */
		}
		if ((i & 0xff) < l)
			printf("\rPass 2 %04x", i & 0xff00);
	}
	printf("\rPass 2 %04x", himark);
}												/*  End of Pass 2 */

/**************************************************************
	Pass three of disassembly. Search for references to
	un-initialized data or split references and, if found,
	generate EQU statements for them.
***************************************************************/

pass3()
{
	int	i, j, l;

	printf("\nPass 3 0000");
	j = 1;
	for (i=0; ; )
	{
		l = i & 0xfff;
		if (pflg[i] & 1)		/* if is un-initialized or is referenced */
		{
			if ((pflg[i] & 8) || !(pflg[i] & 2)) ;
									/* ignore if not referenced or not split opcode */
			else
			{
				if (j)			/* do a newline if first one */
				{
					j = 0;
					fprintf(fp, "\n;");
				}
				fprintf(fp, "\nX%04x\tequ\t", i);		/* do EQU statement */
				puthex(i);
			}
		}
		i++;
		if (!i)
			break;
		if ((i & 0xfff) < l)
			printf("\rPass 3 %04x", i);
	}
	printf("\rPass 3 ffff");
	fprintf(fp, "\n;\n\tend\n;\n\n");
	fflush(fp);
	fclose(fp);
}												/*  End of Pass 3   */

doopcode(char *mnem)
{
	char	c, *ptr;

	ptr = mnem;
	while (c = *ptr++)
	{
		if (c == ' ')
		{
			putc('\t', fp);
			kcnt = (kcnt + 8) & 0x78;
		}
		else
		{
			putc(c, fp);
			kcnt++;
		}
	}
}

splitcheck(int i)				/* test for reference to split opcode */
{
	if (pflg[i] & 8) ;				/* ignore if not referenced */
	else
		pflg[i] = (pflg[i] | 2);	/* else flag split ref */
}

puthex(word j)					/* output hex in ascii to file */
{
	if (j < 10)
		kcnt += fprintf(fp, "%x", j);
	else if (j < 16)
		kcnt += fprintf(fp, "0%xh", j);
	else if (j < 0xa0)
		kcnt += fprintf(fp, "%xh", j);
	else if (j < 0x100)
		kcnt += fprintf(fp, "0%xh", j);
	else if (j < 0xa00)
		kcnt += fprintf(fp, "%xh", j);
	else if (j < 0x1000)
		kcnt += fprintf(fp, "0%xh", j);
	else if (j < 0xa000)
		kcnt += fprintf(fp, "%xh", j);
	else
		kcnt += fprintf(fp, "0%xh", j);
}

ascii(int i)					/* make data printable ascii */
{
	i = i & 0x7f;
	if (i == 0x7f)
		return ('.');
	else if (i < 0x20)
		return ('.');
	else
		return (i);
}

error(char *str1, char *str2)				/* fatal error trap */
{
	printf("\n%s%s", str1, str2);
	exit(0);
}

/* End of DZ80.C */

